Attribute VB_Name = "MdlEquiv"
Option Explicit

Type EquCkt4_t
    sngC0 As Single         '' C0:Equivalent parallel capacitance
    sngC1 As Single         '' C1:Equivalent series capacitance
    sngL1 As Single         '' L1:Equivalent series inductance
    sngR1 As Single         '' R1:Equivalent series resistance
    lngQ As Long            '' Q :Quality factor
    sngCi As Single         '' CI:Crystal impedance
    dblFr As Double         '' fr:Resonant frequency (phase 0 degree):
    dblFs As Double         '' fs:Series resonant frequency (Gmax)
End Type

Public EquData As EquCkt4_t

''Internal local buffer=====================================================
Private fbuf(0 To 1600) As Double
Private gbuf(0 To 1600) As Single, bbuf(0 To 1600) As Single
''==========================================================================

''----------------
''Interpolation calculation
''----------------
Private Function hokan(a As Double, a1 As Double, a2 As Double, b1 As Double, b2 As Double) As Double
    hokan = (a - a1) / (a2 - a1) * (b2 - b1) + b1

End Function

''----------------------------------------------------------------------------
''  Function name  FEquCkt4
''  Function    F[Four elements] Calculating equivalent circuit constant (by specifying an array)
''  Argument    FFc0 As Double     C0 calculated frequency
''                              (If the specified value does not exist, the frequency at the 0 point is applied.)
''            Freq() As Double  Frequency array
''            G() As Single     G (conductance) data array
''            B() As Single     B (susceptance) data array
''            Point As Long     Number of measurement points
''  Return value  FSuccessful (True) / Error (False)
''  Functional descriptionsFCalculates the equivalent circuit constant of four elements using the specified admittance data array, and stores the result in the EquData variable.
''----------------------------------------------------------------------------
Function EquCkt4(Fc0 As Double, Freq() As Double, G() As Single, B() As Single, Point As Long) As Boolean
    Dim phs(0 To 1600) As Single, pi As Single
    Dim pt As Long, pc0 As Long, pgmax As Long
    Dim pfr As Long, pfx As Long
    Dim f1 As Double, f2 As Double
    Dim x1 As Double, x2 As Double, L1 As Double, l2 As Double
    Dim i As Long
    
    EquCkt4 = False

    pt = Point - 1
    
    pi = 4 * Atn(1)         ''
    For i = 0 To pt         ''Phase calculation
        phs(i) = Atn(B(i) / G(i)) * 180 / pi
        If G(i) < 0 Then
            If B(i) < 0 Then
                phs(i) = phs(i) - 180
            Else
                phs(i) = phs(i) + 180
            End If
        End If
    Next i
    
    ''      ************************
    pfr = 0
    For i = 1 To pt
        If phs(i) <= 0 Then pfr = i: Exit For   ''Searches for 0-degree phase
    Next i
    If pfr <= 0 Then Exit Function
    x1 = Freq(pfr - 1): x2 = Freq(pfr)
    L1 = phs(pfr - 1): l2 = phs(pfr)
    EquData.dblFr = hokan(0, L1, l2, x1, x2) '(-l1) / (l2 - l1) * (x2 - x1) + x1
    ''CI      ************************
    L1 = G(pfr - 1): l2 = G(pfr)
    EquData.sngCi = 1 / hokan(EquData.dblFr, x1, x2, L1, l2)
    ''f1      ------------------------
    pfx = 0
    For i = pfr To 0 Step -1
        If phs(i) >= 45 Then pfx = i: Exit For  ''Searches for phase + 45 degrees
    Next i
    If pfx <= 0 Then Exit Function
    x1 = Freq(pfx + 1): x2 = Freq(pfx)
    L1 = phs(pfx + 1): l2 = phs(pfx)
    f1 = hokan(45, L1, l2, x1, x2)
    ''f2      ------------------------
    pfx = pt
    For i = pfr - 1 To pt
        If phs(i) <= -45 Then pfx = i: Exit For ''Searches for phase - 45 degrees
    Next i
    If pfx <= 0 Then Exit Function
    x1 = Freq(pfx - 1): x2 = Freq(pfx)
    L1 = phs(pfx - 1): l2 = phs(pfx)
    f2 = hokan(-45, L1, l2, x1, x2)
    ''Gmax  ------------------------
    pgmax = 0
    For i = 1 To pt
        If G(i) > G(pgmax) Then pgmax = i
    Next i
    ''fs      ************************
    EquData.dblFs = Freq(pgmax)
    ''R1      ************************
    EquData.sngR1 = 1 / G(pgmax)
    ''Q@      ************************
    EquData.lngQ = Abs(EquData.dblFs / (f2 - f1))
    ''L1      ************************
    EquData.sngL1 = Abs(CSng(EquData.lngQ) * EquData.sngR1 / (2 * pi * EquData.dblFs))
    ''C1      ************************
    EquData.sngC1 = 1 / (2 * pi * EquData.dblFs * CSng(EquData.lngQ) * EquData.sngR1)
    ''C0      ************************
    pc0 = 0
    For i = 0 To pt
        If Fc0 = Freq(i) Then pc0 = i: Exit For
    Next i
    EquData.sngC0 = 1 / (2 * pi * Freq(pc0) * (1 / B(pc0))) - EquData.sngC1
    
    EquCkt4 = True

End Function

''----------------------------------------------------------------------------------
''  Function name  FEquCkt4CH
''  Function    F[Four elements] Calculating equivalent circuit constant (by specifying a channel)
''  Argument    FPID As Long       Packet ID (specify value obtained by BisOpenPacket)
''            Fc0 As Double     C0 calculated frequency
''                              (If the specified value does not exist, the frequency at the 0 point is applied.)
''            Z0 As Double      Characteristic impedance Z0
''            Ch As Long        Channels (1 to 4)
''  Return value  FSuccessful (True) / Error (False)
''  Functional descriptionsFAfter performing admittance conversion of the specified channel data, calculates the equivalent circuit constant of four elements and stores the result in the EquData variable.
''  NoteFThe processing speed is slow because the frequency is read and admittance conversion is executed internally each time.
''----------------------------------------------------------------------------------
Function EquCkt4CH(PID As Long, Fc0 As Double, z0 As Double, Ch As Long)
    Dim pt As Long
    
    EquCkt4CH = False
    
    'Frequency table acquisition
    If QrySourFreqTab(PID, Ch, fbuf(0)) <> 0 Then Exit Function
    'Admittance data acquisition
    If GetYTrans(PID, gbuf(), bbuf(), z0, Ch) <> True Then Exit Function
    
    ''Obtains the number of measurement points (-1) by specifying an extremely large frequency value.
    If QryFetcPoin(PID, Ch, CDbl(1000000000#), pt) <> 0 Then Exit Function
    
    EquCkt4CH = EquCkt4(Fc0, fbuf(), gbuf(), bbuf(), pt + 1)

End Function

